const {response, request} = require('express');
const moment = require('moment-timezone');

const { Op, where } = require("sequelize");

const {db} = require('../tables_db/init-models');

const Diary = db.Diary;
const Patient_Cita = db.Patient_Cita;
const Medic = db.Medic;
const ConsultingRoom = db.ConsultingRoom;

/* *************************************************************** */
/* **********************Services GET***************************** */

const getDiary = async (req = request, res = response ) => { 
  const id_company = req.company;
  try {

  const diary = await Diary.findAll({
    where:{
        id_company,
        status_diary: 1     
    },
    include:[
      {
        model: Patient_Cita,
        as: 'patient_cita'
      }
    ]
  });

  return res.json({
    ok: true,
    diary
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getDiaryID = async (req = request, res = response ) => { 
  const id_diary = req.params.id;
  try {

  const diary = await Diary.findByPk(id_diary);
  const patientCita = await Patient_Cita.findOne({
    where:{
      id_diary
    }
  })

  return res.json({
    ok: true,
    diary,
    patientCita
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getPatientCita = async (req = request, res = response ) => { 

  const identity_card = req.params.id;
  
  try {

  const patientCita = await Patient_Cita.findOne({
    where:{
      identity_card
    }
  });

  if(!patientCita){
    return res.status(404).json({
      ok:false,
      msg:'No existe un registro para el número de cédula ingresado.'
    });
  }

  return res.json({
    ok: true,
    patientCita
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};
/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */

const postDiary = async (req = request, res = response ) => { 

  const { body } = req;
  
  body.id_company = req.company;
  body.id_user = req.uid;
  const startDateISO = moment(body.startDate).tz('America/Guayaquil').format('YYYY-MM-DD HH:mm:ss');
  const endDateISO = moment(body.endDate).tz('America/Guayaquil').format('YYYY-MM-DD HH:mm:ss');
  
  try {

    //Validar que no exista otra data igual
    const existData = await Diary.findOne({
      where: {
        [Op.and]:[
          {startDate: {
            [Op.between]: [startDateISO, endDateISO] // Verifica si body.startDate está entre startDate y endDate
          }},
          {id_company: body.id_company},
          {id_consul: body.id_consul},
          {id_doc: body.id_doc},
          {status_diary: 1}         
        ]
      }
    });

    if(existData){
      return res.status(409).json({
        ok: false,
        msg:`No se puede agendar en esta fecha y hora existe cita ya programada desde las ${new Date(existData.startDate).toLocaleTimeString()} hasta las ${new Date(existData.endDate).toLocaleTimeString()}.`
      });
    }

    const newDiary = new Diary( body );
    await newDiary.save();

    const newPatientCita = new Patient_Cita(body);
    newPatientCita.id_diary = newDiary.id_diary;
    await newPatientCita.save();

  return res.json({
    ok: true,
    newDiary,
    msg: 'Cita agendada con éxito.'
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: 'no hay conexión con el servidor.'
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */

const putDiary = async (req = request, res = response ) => { 

  let { id_diary, title, startDate, endDate, id_consul, id_doc } = req.body;

  try {

    //Validar que exista la data para actualizar
    const existData = await Diary.findByPk( id_diary );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe cita en esta fecha y hora para ser actualizada.`
      });
    }

    await existData.update({title, startDate, endDate, id_consul, id_doc});

    return res.json({
      ok: true,
      msg: 'La cita se actualizo con éxito.',
      existData
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */

const confirmationDiary = async (req = request, res = response ) => { 

  const id = req.params.id;

  try {

    //Validar que exista la data para actualizar
    const existData = await Patient_Cita.findOne({
      where:{
        id_diary: id
      }
    });

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe cita en esta fecha y hora para ser confirmada.`
      });
    }

    await existData.update({confirmation: 1});

    return res.json({
      ok: true,
      msg: 'Cita confirmada con éxito.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};
const confirmationReschedule = async (req = request, res = response ) => { 

  const id = req.params.id;

  try {

    //Validar que exista la data para actualizar
    const existData = await Patient_Cita.findByPk(id);

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe cita en esta fecha y hora para ser reagendada.`
      });
    }

    if(existData.confirmation === 0){
      await existData.update({reschedule: 1});
    }

    return res.json({
      ok: true,
      msg: 'Cita Reagendada con éxito.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getDiaryID,
  getDiary,
  getPatientCita,
  postDiary,
  putDiary,
  confirmationDiary,
  confirmationReschedule
}
/* *************************************************************** */
/* *************************************************************** */