const {response, request} = require('express');

const { Op } = require("sequelize");

const {db} = require('../tables_db/init-models');
const Attention = require('../tables_db/Attention');

const Patient = db.Patient;
const Company = db.Company;
const PatientCompany = db.Patient_Company;

/* *************************************************************** */
/* **********************Services GET***************************** */

const getPatient = async (req = request, res = response ) => { 
  const id_company = req.company;
  let {page, limit} = req.query;
  if(!page || !limit){
    page = 1;
    limit=10;
  }
  try {

    const patients = await Patient.findAndCountAll({
      include:[{
        model: Company, 
        as:'companies', 
        where: { id_company },
        through: {
            attributes: [] // Esto es para no incluir los atributos de la tabla intermedia en los resultados
        },
        attributes:[]
      }],
      attributes:[ 'names_patient', 'document_number', 'phone'],
      order: [
          ['names_patient', 'ASC']
      ],
      limit: parseInt(limit),
      offset: (page - 1) * limit
    });

    return res.json({
      ok: true,
      patients
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getPatientByPk = async (req = request, res = response ) => { 
  const document_number = req.params.id;
  try {

    const patient = await Patient.findOne({
      where: {
        document_number
      }
    });

    return res.json({
      ok: true,
      patient
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getPatientByNameID = async (req = request, res = response ) => { 
  const {id} = req.params;
  try {

  const patients = await Patient.findAll({
    where:{
      [Op.or]:[
        {
          names_patient:{
            [Op.like]: `%${id}%`,
          }
        },
        {
          document_number:{
            [Op.like]:`%${id}%`
          }
        }
      ]
    }
  });

  return res.json({
    ok: true,
    patients
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */

const postPatient = async (req = request, res = response ) => { 
  const id_company = req.company;
  const { body } = req;
  body.id_user = req.uid;
  if(!body.photo){
    body.photo = '';
  }
  try {

    //Validar que no exista otra data igual
    const existData = await Patient.findOne({
      include:[{
        model: Company, 
        as:'companies', 
        where: { id_company: id_company},
        through: {
            attributes: [] // Esto es para no incluir los atributos de la tabla intermedia en los resultados
        }
      }],
      where :{
        document_number : body.document_number
    }
    });

    if(existData){
      let { id_sex, id_civil_status, id_blood_typing, id_sexual_orientation, id_gender, id_disability, disability_patient, agreement, names_patient, birthday, email, address_patient, phone, photo } = req.body;
      await existData.update({id_sex, id_civil_status, id_blood_typing, id_sexual_orientation, id_gender, id_disability, disability_patient, agreement, names_patient, birthday, email, address_patient, phone, photo});
      return res.json({
        ok: true,
        msg: 'Información almacenada con éxito.',
        id: existData.id_patient
      });
    }

    const patientData = await Patient.findOne({
      where:{
        document_number : body.document_number
      }
    });
    let patientCompany = new PatientCompany();
    if(patientData){
      patientCompany.id_patient = patientData.id_patient;
      patientCompany.id_company = id_company;
      await patientCompany.save();
      return res.json({
        ok: true,
        msg:`Paciente ${patientData.names_patient}, fue creado con éxito.`,
        id: patientData.id_patient
      });
    }
    const newPatient = new Patient( body );
    await newPatient.save();

    patientCompany.id_patient = newPatient.id_patient;
    patientCompany.id_company = id_company;
    await patientCompany.save();

    return res.json({
      ok: true,
      msg:`Paciente ${newPatient.names_patient}, fue creado con éxito.`,
      id: newPatient.id_patient
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */

const putPatient = async (req = request, res = response ) => { 
  const id_company = req.company;
  const document_number = req.params.id;
  let { id_sex, id_civil_status, id_blood_typing, id_sexual_orientation, id_gender, id_disability, disability_patient, agreement, names_patient, birthday, email, address_patient, phone, photo } = req.body;
  try {

    //Validar que exista la data para actualizar
    const existData = await Patient.findOne({
      include:[{
        model: Company, 
        as:'companies', 
        where: { id_company: id_company},
        through: {
            attributes: [] // Esto es para no incluir los atributos de la tabla intermedia en los resultados
        }
      }],
      where :{
        document_number
      }
    });

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser actualizada.`
      });
    }

    await existData.update({id_sex, id_civil_status, id_blood_typing, id_sexual_orientation, id_gender, id_disability, disability_patient, agreement, names_patient, birthday, email, address_patient, phone, photo});
    return res.json({
      ok: true,
      msg: 'Información actualizada con éxito.',
      existData
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */

const deletePatient = async (req = request, res = response ) => { 

  const { id } = req.params.id;

  try {

    //Validar que exista la data para eliminar
    const existData = await Patient.findByPK( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser eliminada.`
      });
    }

    await existData.update({status_patient: 0});

    return res.json({
      ok: true,
      msg: 'Información eliminada con éxito.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getPatient,
  getPatientByPk,
  getPatientByNameID,
  postPatient,
  putPatient,
  deletePatient
}
/* *************************************************************** */
/* *************************************************************** */