const {response, request} = require('express');

const { Op } = require("sequelize");

const {db} = require('../tables_db/init-models');

const Product = db.Product;
const Company = db.Company;

/* *************************************************************** */
/* **********************Services GET***************************** */

const getProductsByCompany = async (req = request, res = response ) => { 

  try {

  const products = await Product.findAll({
    where: {
      id_company: req.company
    }
  });

  return res.json({
    ok: true,
    products
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getProductsByCompanyLike = async (req = request, res = response ) => { 

  const like = req.params.id

  try {

  const products = await Product.findAll({
    where: {
      id_company: req.company,
      detail:{
        [Op.like]:`%${like}%`
      }
    }
  });

  return res.json({
    ok: true,
    products
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getProductByPk = async (req = request, res = response ) => { 

  const id = req.params.id;
  try {

  const product = await Product.findByPk(id);

  return res.json({
    ok: true,
    product
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */

const postProduct = async (req = request, res = response ) => { 

  const { body } = req;
  body.id_company = req.company;
  try {

    //Validar que no exista otra data igual
    const existData = await Product.findOne({
      where :{
      [Op.and]:[
        {detail : body.detail},
        {id_company: body.id_company}
      ]
    }
    });

    if(existData){
      return res.status(409).json({
        ok: false,
        msg:`Ya existe la información que desea ingresar.`
      });
    }

    const newProduct = new Product( body );
    await newProduct.save();

  return res.json({
    ok: true,
    newProduct
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */

const putProduct = async (req = request, res = response ) => { 

  const { id } = req.params.id;
  let { detail, price, iva } = req.body;

  try {

    //Validar que exista la data para actualizar
    const existData = await Product.findByPK( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser actualizada.`
      });
    }

    await existData.update({detail, price, iva});

    return res.json({
      ok: true,
      msg: 'Información actualizada con éxito.',
      existData
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */

const deleteProduct = async (req = request, res = response ) => { 

  const { id } = req.params.id;

  try {

    //Validar que exista la data para actualizar
    const existData = await Product.findByPK( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser eliminada.`
      });
    }

    await existData.update({status_product: 0});

    return res.json({
      ok: true,
      msg: 'Información eliminada con éxito.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getProductsByCompany,
  getProductsByCompanyLike,
  getProductByPk,
  postProduct,
  putProduct,
  deleteProduct
}
/* *************************************************************** */
/* *************************************************************** */